/*****************************************************************************/

/*
 *
 *   Copyright (c) 2002, Smart Link Ltd.
 *   All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *       1. Redistributions of source code must retain the above copyright
 *          notice, this list of conditions and the following disclaimer.
 *       2. Redistributions in binary form must reproduce the above
 *          copyright notice, this list of conditions and the following
 *          disclaimer in the documentation and/or other materials provided
 *          with the distribution.
 *       3. Neither the name of the Smart Link Ltd. nor the names of its
 *          contributors may be used to endorse or promote products derived
 *          from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/*
 *
 *	mdm_country.c  --  Smart Link Soft Modem Country set definitions.
 *
 *	Author: Sasha K (sashak@smlink.com)
 *
 */

/*****************************************************************************/

#define __KERNEL_SYSCALLS__
#include <linux/kernel.h>
#include <linux/ctype.h>
#include <linux/string.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/unistd.h>
#include <linux/vmalloc.h>
#include <linux/errno.h>
#include <asm/uaccess.h>

#include <linux_compat.h>

#define COUNTRY_NAME_LENGTH 24
#define COUNTRY_INDEX_SIZE  512

struct country_set {
	unsigned id;
	char name[COUNTRY_NAME_LENGTH];
	unsigned char setting[194];
};

struct country_index {
	unsigned id;
	char name[COUNTRY_NAME_LENGTH];
	unsigned offset;
};



/* default country set definitions */
static struct country_set this_country_set = {
	0xb5, "USA",
	{ 36, 64, 1, 6, 50, 150, 43, 0, 255, 65, 1, 255, 10, 80, 15, 35, 18, 60, 4, 18, 60, 0, 10, 0, 5, 1, 0, 1, 120, 0, 0, 1, 1, 0, 1, 0, 5, 50, 2, 65, 1, 255, 2, 65, 1, 255, 10, 0, 3, 0, 1, 255, 1, 25, 1, 2, 0, 0, 5, 43, 0, 255, 10, 43, 50, 43, 0, 43, 0, 0, 0, 0, 0, 1, 4, 30, 25, 50, 250, 250, 1, 1, 0, 6, 0, 80, 2, 2, 0, 19, 0, 100, 60, 5, 0, 0, 0, 53, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 66, 0, 160, 36, 0, 128, 0, 0, 0, 100, 0, 15, 60, 22, 80, 0, 1, 181, 0, 120, 85, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }
};


/* small strcmp */
extern inline int case_insens_strcmp(char *s1, char *s2)
{
	for (; tolower(*s1) == tolower(*s2); s1++, s2++) {
		if (*s1 == '\0')
			break;
	}
	return *s1 - *s2;
}
#ifdef strcmp			/* sometimes macro */
#undef strcmp
#endif
#define strcmp(s1,s2) case_insens_strcmp(s1,s2)

static int errno;

struct country_set *init_country_set(char *country_file, int id, char *name)
{
	mm_segment_t fs;
	struct country_set *country_set = &this_country_set;
	struct country_index *index;
	int fd;

	if ((id < 0 && ( !name || !*name)) ||    /* force default */
	    id == country_set->id ||
	    (id < 0 && !strcmp(country_set->name,name)))
		goto ok;

	fs = get_fs();
	set_fs(get_ds());

	if (!country_file || !*country_file ||
	    (fd = open(country_file, 0, 0)) < 0) {
		printk(KERN_ERR "slmdm: cannot open country file `%s'.\n",
		       country_file?country_file:"");
		goto error;
        }

	index = (struct country_index *)vmalloc(COUNTRY_INDEX_SIZE*sizeof(*index));
	if (!index) {
		printk(KERN_ERR "slmdm: out of mem loading `%s'.\n",
		       country_file);
		goto error1;
	}
		
        if ( read(fd, (void *)index, COUNTRY_INDEX_SIZE*sizeof(*index)) < 0 ) {
                printk(KERN_ERR "slmdm: cannot read file `%s'.\n", country_file);
		goto error2;
        }

	if ( id >= 0 ) { /* search by id */
		if (id >= COUNTRY_INDEX_SIZE || !*index[id].name) {
			printk(KERN_ERR "slmdm: unknown country code %x.\n", id);
			goto error2;
		}
	} else {   /* search by name */
		for ( id = 0 ; id < COUNTRY_INDEX_SIZE ; id++ ) {
			if (!strcmp(index[id].name,name)) {
				break;
			}
		}
		if (id >= COUNTRY_INDEX_SIZE) {
			printk(KERN_ERR "slmdm: unknown country name `%s'.\n", name);
			goto error2;
		}
	}

	country_set->id = id;
	memcpy(country_set->name, index[id].name, sizeof(country_set->name));

	if (lseek(fd, COUNTRY_INDEX_SIZE*sizeof(*index) + index[id].offset, 0) < 0) {
		printk(KERN_ERR "slmdm: cannot seek `%s'.\n", country_file);
		goto error2;
	}

#ifdef DEBUG_COUNTRY
	if (read(fd, country_set, sizeof(*country_set)) < 0) {
#else
	if (read(fd, country_set->setting, sizeof(country_set->setting)) < 0) {
#endif
		printk(KERN_ERR "slmdm: cannot read set from `%s'.\n", country_file);
		goto error2;
	}

#ifdef DEBUG_COUNTRY
	if (country_set->id != id) {
		printk(KERN_ERR "slmdm: bad country id (%x != %x).\n",
		       country_set->id, id);
		goto error2;
	}
#endif
 error2:
	vfree(index);
 error1:
	close(fd);
 error:
	set_fs(fs);
 ok:
	printk(KERN_INFO "slmdm: country set is 0x%x (%s).\n",
	       country_set->id, country_set->name);
	return country_set;
}

